/*
 Automatischer Motorschutz V4

 Modul zur Überwachung der Drehzahl in Abhängigkeit der Länge des Reglersingals. 
 Überschreitet das Reglersignal eine vorgebbare Länge und unterschreitet dabei
 dann die Drehzahl einen vorzugebbenden  Mindestwert, wird der Motor abgestellt. 
 Ist alles im grünen Bereich, wird das Empfängersignal an den Regler durchgereicht.

 Versionshistorie:

 V1:  Ersterstellung

 V2:  Änderung der rpm-Messung. Es wird nur noch während einem PWM-Zyklus gemessen.
      Damit entfällt der Parameter Messzeit.

 V3:  Anlaufroutine eingefügt

 V4:  Lesen Empfängersignal in eigenes Unterprogramm ausgelagert in dem auch 
      der Timout des Empfängersignals berücksichtigt wird.
*/
#define eeprom_schreiben // nach dem ersten Beschreiben eines Arduino ggf. auskommentieren

// #define test  // Serielle Testausgaben, wenn aktiv


#include <Servo.h>    // Bibliothek für Servoansteuerung
#include <EEPROM.h>   // Bibliothek für EEPROM Nutzung

#include "parameter.h"; // Definition der steuernden Programmparameter

Servo sig_regler;     // Einrichten der Servoinstanz

// Pin-Belegungen und -Nutzung
int pin_empf = 3;     // Eingabesignal vom RC-Empfänger
int pin_regler = 4;   // Ausgabesignal für Regler
int pin_rpm = 2;      // Interrupt 0 zur rpm-Messung
int pin_led = 13;     // LED für Zustandsanzeige

// Variablen
int sig_empf;         // Empfängersignal für Motor ein/aus
int led_status;       // für Abfrage LED an oder aus
int aktsig_regler;        // aktuelles Reglersignal
int altsig_regler;        // letztes Reglersignal
volatile int rpm_count;   // Interruptzähler für Drehzahlmessung
float rpm;                // Ergebnis Drehzalmessung
float dauer_rpm;          // Zeitdauer für Drehzahlmessung
float rpm_faktor;         // Hilfsfeld
unsigned long start_rpm;  // Zeitstempel für Beginn Drehzahlmessung
unsigned long start_anlauf; // Empfängersignal zum ersten mal > motor_aus
unsigned long start_zeit;   // Zeit beim Anlaufen des Motors
unsigned long zeit_anlauf;  // Laufzeit nach dem Anlaufen
boolean nothalt;          // bei true = nothalt aktiv
int p_status;             // Status für Programmsteuung
int intervall;

// Konstanten
const unsigned long timeout_pulsein = 30000;  // = 30 msek
const int warten = 1; // Programmstatus warten auf Motorstart
const int anlauf =2;  // Programmstatus = Anlauf
const int lauf = 3;   // Programmstatus = Motor läuft


void setup() {
  nothalt = false;  // Noch kein Nothalt erfolgt
  p_status = warten;  //Programmstatus auf Warten auf Anlauf
  #ifdef test   // Serielle Ausgabe nur für Tests
  Serial.begin (9600); // Serielle Kommunikation starten
  #endif
  pinMode (pin_rpm, INPUT);   // Pin für Drehzahlsensor = Input 
  pinMode (pin_empf, INPUT);  // Pin für Empfänger = Input
  sig_empf = par[motor_aus];
  aktsig_regler = par[motor_aus];
  sig_regler.attach(pin_regler, par[motor_aus], par[motor_max]); // Reglerausgang initialisieren
  // sig_regler.writeMicroseconds( par[motor_aus]);  // Motorsignal erst mal auf aus
  
  attachInterrupt(0,rpm_IR,CHANGE);   // Interrupt 1 an Pin 2 für rpm-Messung aktivieren

    
  #ifdef eeprom_schreiben   
    schreiben_eeprom();   // initiales Schreiben bei neuem Board
  #endif

  lesen_eeprom();   // Parameterstände aus EEPROM lesen
  
 }

void loop() {
  switch (p_status) {
    case warten:
      sub_warten();   // Warten aufs Motor einschalten
      break;
    case anlauf:      
      sub_anlauf();   // Vorgegebene Anlaufzeit ohne Überwachung
      break;
    case lauf:
      intervall = millis() % 500;  // Blinkinterval für Notstopp
      if(nothalt == true && intervall <= 10) digitalWrite(pin_led, !(digitalRead(pin_led)));  //LED toggeln
      sub_lauf();   // Motor läuft und wird überwacht
      break;
    default:
      sub_fehler ();  // hier darf der Ablauf nie landen. Falls doch
                      // wird der Motor ageschaltet
  } //end switch
}

// Warten auf das Gasgeben <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

void sub_warten () {
  sub_empf();  // Empfängersignal lesen
  sig_regler.writeMicroseconds(aktsig_regler); // und zum Regler schicken
  #ifdef test
  Serial.print("W");
  Serial.print(p_status);
  Serial.print("S");
  Serial.println(sig_empf);
  #endif
  if (sig_empf > par[motor_aus]) {  // es wurde Gas gegeben 
    p_status = anlauf;    // Programmstatus auf Anlauf          
    start_anlauf = millis();  // Zeitstempel Beginn Anlauf
  }
}

// Anlaufen ohne Drehzahlüberwachung <<<<<<<<<<<<<<<<<<<<<<<<<<<<<

void sub_anlauf() {
  sub_empf();  // Empfängersignal lesen
  sig_regler.writeMicroseconds(aktsig_regler);   // und an Regler ausgeben
  #ifdef test
  Serial.print("A");
  Serial.print(p_status);
  Serial.print("s");
  Serial.println(sig_empf);
  #endif
  start_zeit = millis() - start_anlauf;   // = bisherige Laufzeit
  if (start_zeit > par[anlauf_zeit]) p_status = lauf;   // dann Überwachung einschalten
}

// Motor läuft und Drehzahl wir überwacht <<<<<<<<<<<<<<<<<<<<<<<<<<<

void sub_lauf() {
  start_rpm = millis();   // Zeitstempel für pm-Messung
  rpm_count = 0;    // Interruptzähler auf 0  
  sub_empf();  // Empfängersignal lesen
  if (aktsig_regler <= par[motor_aus]){  // Motor wurde ausgeschaltet
    p_status = warten;    // dann da capo
    nothalt = false;      // Nothalt zurücksetzen
    digitalWrite(pin_led,LOW);
  }
  #ifdef test
  Serial.print("S:");
  Serial.print(p_status);
  Serial.print("N:");
  Serial.print(nothalt);
  Serial.print("E:");
  Serial.print(sig_empf);
  #endif
  if (aktsig_regler >= par[motor_check] && nothalt == false){  // Signallänge für rpm-Check liegt an?
    digitalWrite(pin_led,HIGH);   // onboard LED einschalten
    rpm_check();    // Drehzahl messen
    if (rpm <= par[rpm_min]){   // Drehzahl kleiner Vorgabe
      aktsig_regler = par[motor_aus];   // Reglersignal = Motor aus
      nothalt = true;   //Nothalt setzen
    }
  }
  else {    // Signallänge < Länge für rpm-Check
    if(nothalt == false) digitalWrite(pin_led,LOW);  // dann LED ausschalten
  }
  if (nothalt == true) aktsig_regler = par[motor_aus];  // Nothalt bereits gesetzt. 
  #ifdef test
  Serial.print("U:");
  Serial.print(rpm);
  Serial.print("R:");
  Serial.println(aktsig_regler);
  #endif
  sig_regler.writeMicroseconds(aktsig_regler);  // Reglersignal ausgeben
}

// Fehler in der Programmsteuerung <<<<<<<<<<<<<<<<<<<<<<<<
void sub_fehler () {
  sig_regler.writeMicroseconds(par[motor_aus]); // Motor abschalten
}

// Lesen Empfängersignal  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
void sub_empf(){
  sig_empf = pulseIn(pin_empf,HIGH,timeout_pulsein);  // Empfängersignal lesen
  if (sig_empf == 0) {  // Null = Timoeut pulseIn
    aktsig_regler = altsig_regler;  // dann letztes Signal übernehmen
  }
  else {
    aktsig_regler = sig_empf;   // aktuelles Signal übernehmen
    altsig_regler = sig_empf;   // aktuelles Signal = altes Signal
  }   
}

// rpm_check <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
void rpm_check () {
  dauer_rpm = millis() - start_rpm;   // exakte Messdauer
  rpm_faktor = 1000 / dauer_rpm;  // da Messung in millis
  rpm = rpm_count * rpm_faktor; // = Impulse pro Sekunde
  rpm = rpm * 60;               // = Impulse pro Minute
  rpm = rpm / ( par[pole] + 2); // = Drehzahl 
} // end rpm_check

// rpm Interrupt <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
void rpm_IR() {
  rpm_count ++; //Impulse rpm-Sensor + 1
} // end rpm_IR
